﻿/******************************************************************************
 * DISCLAIMER
 * This software is supplied by Renesas Electronics Corporation and is only
 * intended for use with Renesas products. No other uses are authorized. This
 * software is owned by Renesas Electronics Corporation and is protected under
 * all applicable laws, including copyright laws.
 * THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
 * THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
 * LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
 * TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
 * ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
 * FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
 * ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
 * BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * Renesas reserves the right, without notice, to make changes to this
 * software and to discontinue the availability of this software. By using this
 * software, you agree to the additional terms and conditions found by
 * accessing the following link:
 * http://www.renesas.com/disclaimer
 *
 * Copyright (C) 2021 Renesas Electronics Corporation. All rights reserved.
 *****************************************************************************/

using Microsoft.Win32;
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Windows;
using System.Windows.Input;
using Windows.Devices.Bluetooth;
using Windows.Devices.Bluetooth.Advertisement;
using Windows.Devices.Bluetooth.GenericAttributeProfile;
using Windows.Security.Cryptography;
using Windows.Storage.Streams;

//-----------------------------------------------------------------------------
//	FLASH DATA MAP
//-----------------------------------------------------------------------------
//サーミスタ温度校正データ(9byte)
[StructLayout(LayoutKind.Sequential, Pack = 1)]
struct tTHCAL
{
    public short jig;              //2	冶具温度
    public short thout;                //2	PCB外付けのTH温度データ
    public short th;                   //2	PCBのTH温度データ
    public short tp;                   //2	PCBのTP温度データ(offset)
    public byte done;				//1	bit0=1で校正済み
}

//黑体校正データ(7byte)
[StructLayout(LayoutKind.Sequential, Pack = 1)]
struct tBBCAL
{
    public short slp;               //2	黑体**.**℃のTP傾き
    public short tp;                    //2	黑体**.**℃のTP値
    public short th;                    //2	黑体**.**℃のTP値実際の筒の０．０１℃温度
    public byte done;				//1	bit0=1で校正済み
}

//蓋したとき（温度差０℃）のoffset(25℃のoffset)
[StructLayout(LayoutKind.Sequential, Pack = 1)]
struct tOFST
{
    public short tp;                //2	TPoffset
    public short th;                //2	THoffset
    public byte done;				//1	bit0=1で校正済み
}

//製造番号
[StructLayout(LayoutKind.Sequential, Pack = 1)]
struct tSN
{
    public ushort sno;              //2  製品情報：シリアルNo(0x0001-0x0FFF)
    public byte lot;				//1  製品情報：ロットNo(0x01-0xFF)
}

[StructLayout(LayoutKind.Sequential, Pack = 1)]
//public修飾子が付いたものは構造体の外部からアクセスできます．
struct tCFG
{
    public byte tadj_done;      //1  微調整完了フラグ
    public short tadj;          //2  微調整係数(10000倍)
    public tTHCAL hi;               //9  サーミスタ温度校正データ：高温
    public tTHCAL lo;               //9  サーミスタ温度校正データ：低温
    public tOFST ofst;          //5  蓋したとき（温度差０℃）のoffset
    public tBBCAL bb35;       //35 黑体校正データ
    public tBBCAL bb40;       //35 黑体校正データ
    public tBBCAL bb45;       //35 黑体校正データ
    public tBBCAL bb50;       //35 黑体校正データ
    public tBBCAL bb55;       //35 黑体校正データ
    public tSN sn;              //3  製品情報：シリアルNo
    public byte flag;           //1
}


namespace Virtual_UART_Client
{
    enum rcvmod
    {
        RCVMOD_DATA = 0,
        RCVMOD_CFG
    }

    public partial class MainWindow : Window
    {
static byte[] rxbuf = new byte[80];
static int iRcvNum = -1;
static rcvmod ucRcvMode;
static tCFG CFG;

        /// <summary>
        /// RL78/G1D Virtual UART Service UUIDs
        /// </summary>
        static Guid UUID_VUART_SERVICE = new Guid("49535343-FE7D-4AE5-8FA9-9FAFD205E455");
        static Guid UUID_VUART_INDICATION = new Guid("49535343-1E4D-4BD9-BA61-23C647249616");
        static Guid UUID_VUART_WRITE = new Guid("49535343-8841-43F4-A8D4-ECBE34729BB3");
        private const string DEVNAME_RBLE = "Dual-SPP";

        private const int VUART_WRITE_LENGTH = 20;

        // Variable of BluetoothLEAdvertisementWatcher object.
        private BluetoothLEAdvertisementWatcher advertisementWatcher;

        // Variable of BluetoothLEAdvertisementWatcher object.
        private AdvertisementWatcherInformation myDevice;

        // Variable of BluetoothDevice object identified by the specified DeviceId.
        private BluetoothLEDevice bluetoothLeConnectedDevice;

        // Variables of Service and Characteristics. 
        private GattDeviceService gattPrimaryService;
        private GattCharacteristic characteristicIndicate;
        private GattCharacteristic characteristicWrite;

        // Flags. 
        bool isFoundVuartDevice;

        // Variable of notify the control of data changes.
        private ViewModel viewModel;

        // Variable of Timer object.
        private Timer timer;

        // Error number.
        public enum ConnectionProcessStatus
        {
            Success = 0,
            PreprocessingError = 1,
            ServiceError = 2,
            CharacteristicError = 3,
            CCCDError = 4,
        }

        //*********************************************************************
        //アプリケーション処理が記述されたクラス。
        //*********************************************************************
        public MainWindow()
        {
            InitializeComponent();

            // Create an instance of ViewModel.
            // Notify Buttons and TextBoxes of data changes.
            viewModel = new ViewModel();
            // Set viewModel to DataContext property.
            DataContext = viewModel;

            myDevice = new AdvertisementWatcherInformation();   //接続対象評価ボードのアドバタイジングから得た情報を保存するクラス。
            CFG = new tCFG();
        }

        // =============================================================================
        //  CONNECTボタン
        //      Start scanning and find the UUID of the evaluation board
        //      from the received advertising data.
        // =============================================================================
        /// <summary>
        /// Clicking event of Connect button.
        /// </summary>
        /// <param name="sender">The object where the event handler is attached.</param>
        /// <param name="e">The event data.</param>
        private void ButtonConnect_Click(object sender, RoutedEventArgs e)
        {
            viewModel.button_connect_isenabled = false;

            if (bluetoothLeConnectedDevice == null)
            {
                // Not connected.

                viewModel.textbox_status = "Scanning.";
                isFoundVuartDevice = false;

                // Target device information clear.
                myDevice.Clear();

                // Start AdvertisementWatcher.
                AdvertisementWatcher_Start(5000);
            }
            else
            {
                // Connected.

                viewModel.textbox_status = "Disconnecting.";
                viewModel.button_log_isenabled = false;
                viewModel.button_file_isenabled = false;
                viewModel.button_send_isenabled = false;
#if true
                //IKE:ADD
                SendDco();
#endif
                Disconnect();
            }
        }

        // =============================================================================
        //  BluetoothLEAdvertisementWatcher process.
        //      Start scanning and find the UUID of the evaluation board
        //      from the received advertising data.
        // =============================================================================

        /// <summary>
        /// BluetoothLEAdvertisementWatcher start process.
        /// It is executed when pairing.
        /// </summary>
        /// <param name="timeout">Time to run BluetoothLEAdvertisementWatcher. Time unit is millisecond.</param>
        private void AdvertisementWatcher_Start(int timeout)
        {
            Debug.WriteLine("AdvertisementWatcher Start.");

            advertisementWatcher = new BluetoothLEAdvertisementWatcher();
            advertisementWatcher.ScanningMode = BluetoothLEScanningMode.Active;
            advertisementWatcher.Received += AdvertisementWatcher_Received;
            advertisementWatcher.Start();

            // Start timetout timer.
            Debug.WriteLine("timer start.");
            var timerCallbadk = new TimerCallback(AdvertisementWatcherTimer_Callback);
            timer = new Timer(timerCallbadk, null, timeout, 0);
        }

        /// <summary>
        /// BluetoothLEAdvertisementWatcher stop process
        /// </summary>
        private void AdvertisementWatcher_Stop()
        {
            // Stop timeout timer.
            Debug.WriteLine("timer stop.");
            timer.Change(Timeout.Infinite, Timeout.Infinite);
            timer.Dispose();
            timer = null;

            Debug.WriteLine("AdvertisementWatcher Stop.");
            advertisementWatcher.Stop();
            advertisementWatcher.Received -= AdvertisementWatcher_Received;
            advertisementWatcher = null;

            if (isFoundVuartDevice == true)
            {
                Connect();
            }
        }

        /// <summary>
        /// Event handler of advertising reception.
        /// </summary>
        /// <param name="watcher">An object to receive Bluetooth Low Energy (LE) advertisements.</param>
        /// <param name="eventArgs">Provides data for a Received event on a BluetoothLEAdvertisementWatcher.</param>
        public void AdvertisementWatcher_Received(BluetoothLEAdvertisementWatcher watcher,
                                                  BluetoothLEAdvertisementReceivedEventArgs eventArgs)
        {
            Debug.WriteLine($"Adv received. {eventArgs.BluetoothAddress.ToString("X12")}, {eventArgs.Advertisement.LocalName}");

            if (advertisementWatcher != null)
            {
                // Judge device name of evaluation board.
                if (eventArgs.Advertisement.LocalName.Contains(DEVNAME_RBLE) == true)
                {
                    Debug.WriteLine("Found Peripheral Device.");

                    myDevice.BluetoothDeviceAddrss_string = String.Format("{0:X}", eventArgs.BluetoothAddress);
                    myDevice.Name = eventArgs.Advertisement.LocalName;
                    myDevice.BluetoothDeviceAddrss_ulong = eventArgs.BluetoothAddress;
                    Debug.WriteLine($"BDA ={myDevice.BluetoothDeviceAddrss_string}");
                    Debug.WriteLine($"Name={myDevice.Name}");
                    Debug.WriteLine($"UUID={myDevice.ServiceUuid}");

                    isFoundVuartDevice = true;

                    AdvertisementWatcher_Stop();

                    //DeviceWatcher_Start(7000);
                }
            }
        }

        /// <summary>
        /// Timer callback of BluetoothLEAdvertisementWatcher.
        /// This process runs when the peripheral device is not found.
        /// </summary>
        /// <param name="state">An object that contains application specific information, or null.</param>
        private void AdvertisementWatcherTimer_Callback(object state)
        {
            timer.Change(Timeout.Infinite, Timeout.Infinite);

            AdvertisementWatcher_Stop();

            Debug.WriteLine("Peripheral device was not found.");
            viewModel.textbox_status = "Peripheral device was not found.";
            viewModel.button_connect_isenabled = true;
        }


        // =============================================================================
        //  Connect and Disconnnect process.
        // =============================================================================

        /// <summary>
        /// Connect process.
        /// Search for services and characteristics.
        /// </summary>
        private async void Connect()
        {
            var error = ConnectionProcessStatus.Success;

            viewModel.textbox_status = "Connecting.";

            bluetoothLeConnectedDevice = 
                await BluetoothLEDevice.FromBluetoothAddressAsync(myDevice.BluetoothDeviceAddrss_ulong);
            if (bluetoothLeConnectedDevice == null)
            {
                Debug.WriteLine("BluetoothLEDevice.FromBluetoothAddressAsync error!");
                error = ConnectionProcessStatus.PreprocessingError;
            }
            else
            {
                // Connected to target device.
                Debug.WriteLine("BluetoothLEDevice.FromBluetoothAddressAsync OK.");

                // RL78/G1D Virtual UART Service Structure
                // -------------------------------------------------------------------------------------
                // Type            UUID (Devine name)                                           Property
                // -------------------------------------------------------------------------------------
                // Primary Service D68C0001-A21B-11E5-8CB8-0002A5D5C51B (UUID_VUART_SERVICE)    -
                // Characteristic  D68C0002-A21B-11E5-8CB8-0002A5D5C51B (UUID_VUART_INDICATION) Indicate
                // Characteristic  D68C0003-A21B-11E5-8CB8-0002A5D5C51B (UUID_VUART_WRITE)      Write
                // -------------------------------------------------------------------------------------

                //--------------------------------------------------
                // Search for primary service.
                //--------------------------------------------------
                var serviceResult = await bluetoothLeConnectedDevice.GetGattServicesForUuidAsync(UUID_VUART_SERVICE);
                if (serviceResult.Status == GattCommunicationStatus.Success)
                {

                    Debug.WriteLine("Found primary service");
                    Debug.WriteLine($"servicess={serviceResult.Services.Count}");
                    gattPrimaryService = serviceResult.Services[0];

                    //--------------------------------------------------
                    // Search for indicate characteristic.
                    //--------------------------------------------------
                    var indicationCharacteristicsResult =
                        await serviceResult.Services[0].GetCharacteristicsForUuidAsync(UUID_VUART_INDICATION);
                    if (indicationCharacteristicsResult.Status == GattCommunicationStatus.Success)
                    {
                        Debug.WriteLine("Found indicate characteristic");

                        // Save indicate characteristic.
                        characteristicIndicate = indicationCharacteristicsResult.Characteristics[0];
                        // Register indicate reception event.
                        characteristicIndicate.ValueChanged += ReceiveIndication;

                        //--------------------------------------------------
                        // Search for write characteristic.
                        //--------------------------------------------------
                        var writeCharacteristicsResult =
                            await serviceResult.Services[0].GetCharacteristicsForUuidAsync(UUID_VUART_WRITE);
                        if (writeCharacteristicsResult.Status == GattCommunicationStatus.Success)
                        {
                            Debug.WriteLine("Found write characteristic");
                            // Save write characteristic.
                            characteristicWrite = writeCharacteristicsResult.Characteristics[0];
                        }
                        else
                        {
                            //error
                            Debug.WriteLine("Error: not found wirte characterisitc");
                        }

                        try
                        {
                            Debug.WriteLine("Write enable indication to CCCD");

                            //--------------------------------------------------
                            // Write indication enable to CCCD
                            //--------------------------------------------------
                            GattCommunicationStatus status =
                                await characteristicIndicate.WriteClientCharacteristicConfigurationDescriptorAsync(GattClientCharacteristicConfigurationDescriptorValue.Notify);// .Indicate);
                            if (status == GattCommunicationStatus.Success)
                            {
                                //--------------------------------------------------
                                // Connection completed.
                                //--------------------------------------------------

                                Debug.WriteLine($"Connected to {bluetoothLeConnectedDevice.Name}.");
                                viewModel.textbox_status = "Connected.";
                                viewModel.button_connect_content = "Disconnect";
                                viewModel.textbox_bdaddress = myDevice.BluetoothDeviceAddrss_string;

                                // Register BluetoothLEDevice.ConnectionStatusChanged event.
                                // To receive disconnection from peripheral.
                                bluetoothLeConnectedDevice.ConnectionStatusChanged += ConnectionStatusChanged;

                                viewModel.button_connect_isenabled = true;
//IKE:DEL                                viewModel.button_log_isenabled = true;
//IKE:DEL                                viewModel.button_file_isenabled = true;
                                viewModel.button_send_isenabled = true;
                            }
                            else
                            {
                                Debug.WriteLine("Error: failed to write to CCCD");
                                viewModel.textbox_status = "Failed to write to CCCD. Please disconnect.";
                                error = ConnectionProcessStatus.CCCDError;
                            }
                        }
                        catch (Exception ex)
                        {
                            MessageBox.Show(ex.ToString(), "CCCD Write Error.");
                        }
                    }
                    else
                    {
                        // error
                        Debug.WriteLine("Error: not found indicate characterisitc");
                        error = ConnectionProcessStatus.CharacteristicError;
                    }
                }
                else
                {
                    // Timeout.
                    // Search for primary service failed.
                    Debug.WriteLine($"GetGattServicesForUuidAsync err = {serviceResult.Status}");
                    bluetoothLeConnectedDevice.Dispose();
                    bluetoothLeConnectedDevice = null;
                    error = ConnectionProcessStatus.ServiceError;
                }
            }

            if ((error == ConnectionProcessStatus.PreprocessingError) ||
                (error == ConnectionProcessStatus.ServiceError))
            {
                // FromIdAsync error or GetGattServicesForUuidAsync error
                viewModel.textbox_status = "Connection failed.";
                viewModel.button_connect_isenabled = true;
            }
            else if (error != ConnectionProcessStatus.Success)
            {
                Disconnect();
            }
            else
            {
                // do nothing.
            }
        }

        /// <summary>
        /// Disconnect process.
        /// </summary>
        private void Disconnect()
        {
            Debug.WriteLine("Disconnect.");

            if (characteristicIndicate != null)
            {
                characteristicIndicate.Service.Dispose();
            }
            if (gattPrimaryService != null)
            {
                gattPrimaryService.Dispose();
            }
            if (bluetoothLeConnectedDevice != null)
            {
                bluetoothLeConnectedDevice.Dispose();
            }
        }

        /// <summary>
        /// Event handler that occurs when the connection status changes.
        /// Used by this application to determine if the disconnect is complete.
        /// </summary>
        /// <param name="sender">Event source of BluetoothLEDevice.</param>
        /// <param name="e">The event data. If there is no event data, this parameter will be null.</param>
        private void ConnectionStatusChanged(BluetoothLEDevice sender, object e)
        {
            Debug.WriteLine("ConnectionStatusChanged={0}", sender.ConnectionStatus);

            if (sender.ConnectionStatus == BluetoothConnectionStatus.Disconnected)
            {
                // Disconnect and clear value.
                if (bluetoothLeConnectedDevice != null)
                {
                    bluetoothLeConnectedDevice.ConnectionStatusChanged -= ConnectionStatusChanged;
                    bluetoothLeConnectedDevice = null;
                }
                if (gattPrimaryService != null)
                {
                    gattPrimaryService = null;
                    characteristicIndicate.ValueChanged -= ReceiveIndication;
                    characteristicIndicate = null;
                    characteristicWrite = null;
                }

                viewModel.textbox_status = "Disconnected.";
                viewModel.button_connect_content = "Connect";

                viewModel.button_connect_isenabled = true;
                viewModel.button_log_isenabled = false;
                viewModel.button_file_isenabled = false;
                viewModel.button_send_isenabled = false;
            }
        }


        // =============================================================================
        //  Data reception and sending process.
        // =============================================================================

        /// <summary>
        /// Event handler that is called when a indication is received.
        /// </summary>
        private void ReceiveIndication(GattCharacteristic sender, GattValueChangedEventArgs eventArgs)
        {
            string str = "";
            int i;
            byte[] data = new byte[eventArgs.CharacteristicValue.Length];

            DataReader reader = DataReader.FromBuffer(eventArgs.CharacteristicValue);
            reader.ReadBytes(data);

            for (i = 0; i < eventArgs.CharacteristicValue.Length; i++)
            {
                if (iRcvNum >= 0)
                {
                    //格納中
                    rxbuf[iRcvNum] = data[i];
                    ++iRcvNum;
                    if (ucRcvMode==rcvmod.RCVMOD_DATA)
                    {
                        if (iRcvNum > 1)
                        {
                            if ((rxbuf[iRcvNum - 2] == 0x0D) && (rxbuf[iRcvNum - 1] == 0x0A))
                            {
                                //"\r\n"の前まで文字列化する。ファイル出力では改行してくれるから。
                                str = System.Text.Encoding.ASCII.GetString(rxbuf, 0, (iRcvNum-2));  //byte配列からstring型に変換
                                {
                                    string stPath = string.Format("CFG_{0}.TXT", myDevice.BluetoothDeviceAddrss_string);
                                    string stCurrentDir = System.IO.Directory.GetCurrentDirectory(); // カレントディレクトリを取得

                                    using (StreamWriter outputFile = new StreamWriter(Path.Combine(stCurrentDir, stPath), true))
                                    {
                                        outputFile.WriteLine(str);
                                    }
                                }
                                //"\r\n"を追加する。TEXTBOXへの出力では改行が必用だから。
                                str += "\r\n";
                                Debug.WriteLine(str);
                                iRcvNum = -1;   //初めから
                            }
                        }
                    }
                    else if (ucRcvMode == rcvmod.RCVMOD_CFG)
                    {
                        if (iRcvNum >= Marshal.SizeOf(CFG))
                        {
                            //rxbufのポインタを取得
                            GCHandle gch = GCHandle.Alloc(rxbuf, GCHandleType.Pinned);
                            IntPtr ptr = gch.AddrOfPinnedObject();
                            //rxbufのコピー
                            CFG = (tCFG)Marshal.PtrToStructure(ptr, typeof(tCFG));
                            gch.Free();
                            iRcvNum = -1;   //初めから
                            OutCfg();
                            MessageBox.Show("Recv [FD]", "");
                        }

                    }
                }
                else {
                    if (i > 0)
                    {
                        if (data[i - 1] == 0xAD)
                        {
                            if (data[i] == 0x03)
                            {
                                //データパケット
                                ucRcvMode = rcvmod.RCVMOD_DATA;
                                iRcvNum = 0;    //格納開始
                                ++i;    //サイズ
                            }
                            else if (data[i] == 0x02)
                            {
                                //切断通知
                            }
                            else if (data[i] == 0x7F)
                            {
                                //CFG
                                ucRcvMode = rcvmod.RCVMOD_CFG;
                                iRcvNum = 0;    //格納開始
                            }
                        }
                    }
                }
            }
            if (!str.Equals("")) {
	            this.Dispatcher.Invoke((Action)(() =>
	            {
	                // Scroll to the end of the string and then add the string.
	                TextBoxReceive.ScrollToEnd();
	                viewModel.textbox_receive_text += str;
	            }));
            }
        }


#if true
//IKE ADD
        private async void SendDco()
        {
			//DISCONNECTを送信
			byte[] txbuf = new byte[2];
			txbuf[0] = (byte)0xAD;
			txbuf[1] = (byte)0x02;

			// Create a buffer from the byte array.
			IBuffer ibuffer = CryptographicBuffer.CreateFromByteArray(txbuf);
			await characteristicWrite.WriteValueAsync(ibuffer, GattWriteOption.WriteWithoutResponse);
        }
        private async void SendCfg()
        {
            //CFGコマンドを送信
            byte[] txbuf = new byte[2];
            txbuf[0] = (byte)0xAD;
            txbuf[1] = (byte)0x7F;

            // Create a buffer from the byte array.
            IBuffer ibuffer = CryptographicBuffer.CreateFromByteArray(txbuf);
            await characteristicWrite.WriteValueAsync(ibuffer, GattWriteOption.WriteWithoutResponse);
            //MessageBox.Show("Send [Read FD]", "");
        }

        private void OutCfg()
        {
            //const string FNAME_LOG = "CFG.TXT";
            string str = "";
            string stPath = string.Format("CFG_{0}.TXT",myDevice.BluetoothDeviceAddrss_string);
            string stCurrentDir = System.IO.Directory.GetCurrentDirectory(); // カレントディレクトリを取得

            using (StreamWriter outputFile = new StreamWriter(Path.Combine(stCurrentDir, stPath), true))
            {
                str = string.Format("DEVICE,[{0}]", myDevice.BluetoothDeviceAddrss_string); outputFile.WriteLine(str);
                str = string.Format("CFG.tadj_done = {0}", CFG.tadj_done); outputFile.WriteLine(str);
                str = string.Format("CFG.tadj = {0}", CFG.tadj); outputFile.WriteLine(str);
                str = string.Format("CFG.hi.jig = {0}", CFG.hi.jig); outputFile.WriteLine(str);
                str = string.Format("CFG.hi.thout = {0}", CFG.hi.thout); outputFile.WriteLine(str);
                str = string.Format("CFG.hi.th = {0}", CFG.hi.th); outputFile.WriteLine(str);
                str = string.Format("CFG.hi.tp = {0}", CFG.hi.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.hi.done = {0}", CFG.hi.done); outputFile.WriteLine(str);
                str = string.Format("CFG.lo.jig = {0}", CFG.lo.jig); outputFile.WriteLine(str);
                str = string.Format("CFG.lo.thout = {0}", CFG.lo.thout); outputFile.WriteLine(str);
                str = string.Format("CFG.lo.th = {0}", CFG.lo.th); outputFile.WriteLine(str);
                str = string.Format("CFG.lo.tp = {0}", CFG.lo.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.lo.done = {0}", CFG.lo.done); outputFile.WriteLine(str);
                str = string.Format("CFG.ofst.tp = {0}", CFG.ofst.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.ofst.th = {0}", CFG.ofst.th); outputFile.WriteLine(str);
                str = string.Format("CFG.ofst.done = {0}", CFG.ofst.done); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[0].slp = {0}", CFG.bb35.slp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[0].tp = {0}", CFG.bb35.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[0].th = {0}", CFG.bb35.th); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[0].done = {0}", CFG.bb35.done); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[1].slp = {0}", CFG.bb40.slp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[1].tp = {0}", CFG.bb40.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[1].th = {0}", CFG.bb40.th); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[1].done = {0}", CFG.bb40.done); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[2].slp = {0}", CFG.bb45.slp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[2].tp = {0}", CFG.bb45.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[2].th = {0}", CFG.bb45.th); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[2].done = {0}", CFG.bb45.done); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[3].slp = {0}", CFG.bb50.slp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[3].tp = {0}", CFG.bb50.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[3].th = {0}", CFG.bb50.th); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[3].done = {0}", CFG.bb50.done); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[4].slp = {0}", CFG.bb55.slp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[4].tp = {0}", CFG.bb55.tp); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[4].th = {0}", CFG.bb55.th); outputFile.WriteLine(str);
                str = string.Format("CFG.bb[4].done = {0}", CFG.bb55.done); outputFile.WriteLine(str);
                str = string.Format("CFG.sn.sno = 0x{0:X4}", CFG.sn.sno); outputFile.WriteLine(str);
                str = string.Format("CFG.sn.lot = 0x{0:X2}", CFG.sn.lot); outputFile.WriteLine(str);
                str = string.Format("CFG.flag = 0x{0:X2}", CFG.flag); outputFile.WriteLine(str);
            }
        }

#endif


        //IKE ADD
        private void Button_ReadFD_Click(object sender, RoutedEventArgs e)
        {
            //CFGコマンドを送信
            SendCfg();
        }
    }

    /// <summary>
    /// Advertising data of peripheral device.
    /// </summary>
    public class AdvertisementWatcherInformation
    {
        public string BluetoothDeviceAddrss_string { get; set; }
        public string Name { get; set; }
        public Guid ServiceUuid { get; set; }
        public ulong BluetoothDeviceAddrss_ulong { get; set; }

        public void Clear()
        {
            BluetoothDeviceAddrss_string = null;
            Name = null;
            ServiceUuid = new Guid("00000000-0000-0000-0000-000000000000");
            BluetoothDeviceAddrss_ulong = 0;
        }
    }


    /// <summary>
    /// Notifies the control of data updates and updates the control's display.
    /// </summary>
    public class ViewModel : INotifyPropertyChanged
    {
        public event PropertyChangedEventHandler PropertyChanged;

        public ViewModel()
        {
            //ここが初期化
            textbox_status = "Not connected.";
            textbox_bdaddress = "";
            button_connect_content = "Connect";
            button_log_content = "Log";

            button_connect_isenabled = true;
            button_log_isenabled = false;
            button_file_isenabled = false;
            button_send_isenabled = false;
        }

        /// <summary>
        /// Property : TextBox.Text of connection status.
        /// </summary>
        private string _textbox_status;
        public string textbox_status
        {
            get { return _textbox_status; }
            set { _textbox_status = value; NotifyPropertyChanged("textbox_status"); }
        }

        /// <summary>
        /// Property : TextBox.Text of Bluetooth Device Address.
        /// </summary>
        private string _textbox_bdaddress;
        public string textbox_bdaddress
        {
            get { return _textbox_bdaddress; }
            set { _textbox_bdaddress = value; NotifyPropertyChanged("textbox_bdaddress"); }
        }

        /// <summary>
        /// Property : TextBox.Text of receive data. Display receive data.
        /// </summary>
        private string _textbox_receive_text;
        public string textbox_receive_text
        {
            get { return _textbox_receive_text; }
            set { _textbox_receive_text = value; NotifyPropertyChanged("textbox_receive_text"); }
        }

        /// <summary>
        /// Property : Button.Content of Connect/Disconnect Button.
        /// </summary>
        private string _button_connect_content;
        public string button_connect_content
        {
            get { return _button_connect_content; }
            set { _button_connect_content = value; NotifyPropertyChanged("button_connect_content"); }
        }

        /// <summary>
        /// Property : Button.Content of Log Button.
        /// </summary>
        private string _button_log_content;
        public string button_log_content
        {
            get { return _button_log_content; }
            set { _button_log_content = value; NotifyPropertyChanged("button_log_content"); }
        }

        /// <summary>
        /// Property : Button.IsEnabled of Connect/Disconnect Button.
        /// </summary>
        private bool _button_connect_isenabled;
        public bool button_connect_isenabled
        {
            get { return _button_connect_isenabled; }
            set { _button_connect_isenabled = value; NotifyPropertyChanged("button_connect_isenabled"); }
        }

        /// <summary>
        /// Property : Button.IsEnabled of Log Button.
        /// </summary>
        private bool _button_log_isenabled;
        public bool button_log_isenabled
        {
            get { return _button_log_isenabled; }
            set { _button_log_isenabled = value; NotifyPropertyChanged("button_log_isenabled"); }
        }

        /// <summary>
        /// Property : Button.IsEnabled of File Button.
        /// </summary>
        private bool _button_file_isenabled;
        public bool button_file_isenabled
        {
            get { return _button_file_isenabled; }
            set { _button_file_isenabled = value; NotifyPropertyChanged("button_file_isenabled"); }
        }

        /// <summary>
        /// Property : Button.IsEnabled of Send Button.
        /// </summary>
        private bool _button_send_isenabled;
        public bool button_send_isenabled
        {
            get { return _button_send_isenabled; }
            set { _button_send_isenabled = value; NotifyPropertyChanged("button_send_isenabled"); }
        }

        private void NotifyPropertyChanged(string propName)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(propName));
            }
        }
    }
}
